<?php
/**
 * Plugin Name: Lightweight GDPR Consent
 * Plugin URI: https://gizzapp.com
 * Description: A lightweight GDPR cookie consent banner with customizable branding
 * Version: 1.0.0
 * Author: John McCormack
 * License: GPL v2 or later
 * Text Domain: lightweight-gdpr
 */

if (!defined('ABSPATH')) exit;

class Lightweight_GDPR_Consent {
    
    private $options;
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'settings_init'));
        add_action('wp_footer', array($this, 'display_consent_banner'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_ajax_lgdpr_accept_cookies', array($this, 'handle_accept'));
        add_action('wp_ajax_nopriv_lgdpr_accept_cookies', array($this, 'handle_accept'));
    }
    
    public function add_admin_menu() {
        add_options_page(
            'GDPR Consent Settings',
            'GDPR Consent',
            'manage_options',
            'lightweight-gdpr',
            array($this, 'settings_page')
        );
    }
    
    public function settings_init() {
        register_setting('lgdpr_settings', 'lgdpr_options');
        
        add_settings_section(
            'lgdpr_section',
            'Cookie Consent Settings',
            array($this, 'section_callback'),
            'lgdpr_settings'
        );
        
        add_settings_field(
            'lgdpr_message',
            'Consent Message',
            array($this, 'message_field'),
            'lgdpr_settings',
            'lgdpr_section'
        );
        
        add_settings_field(
            'lgdpr_logo',
            'Brand Logo URL',
            array($this, 'logo_field'),
            'lgdpr_settings',
            'lgdpr_section'
        );
        
        add_settings_field(
            'lgdpr_primary_color',
            'Primary Color',
            array($this, 'color_field'),
            'lgdpr_settings',
            'lgdpr_section'
        );
        
        add_settings_field(
            'lgdpr_privacy_url',
            'Privacy Policy URL',
            array($this, 'privacy_url_field'),
            'lgdpr_settings',
            'lgdpr_section'
        );
    }
    
    public function section_callback() {
        echo '<p>Configure your GDPR cookie consent banner</p>';
    }
    
    public function message_field() {
        $options = get_option('lgdpr_options');
        $value = isset($options['message']) ? $options['message'] : 'We use cookies to enhance your experience. By continuing to visit this site you agree to our use of cookies.';
        echo '<textarea name="lgdpr_options[message]" rows="3" cols="50" class="large-text">' . esc_textarea($value) . '</textarea>';
    }
    
    public function logo_field() {
        $options = get_option('lgdpr_options');
        $value = isset($options['logo']) ? $options['logo'] : '';
        echo '<input type="text" name="lgdpr_options[logo]" value="' . esc_attr($value) . '" class="regular-text" placeholder="https://example.com/logo.png">';
        echo '<p class="description">Enter the full URL to your logo image</p>';
    }
    
    public function color_field() {
        $options = get_option('lgdpr_options');
        $value = isset($options['color']) ? $options['color'] : '#0073aa';
        echo '<input type="color" name="lgdpr_options[color]" value="' . esc_attr($value) . '">';
    }
    
    public function privacy_url_field() {
        $options = get_option('lgdpr_options');
        $value = isset($options['privacy_url']) ? $options['privacy_url'] : '';
        echo '<input type="text" name="lgdpr_options[privacy_url]" value="' . esc_attr($value) . '" class="regular-text" placeholder="/privacy-policy">';
        echo '<p class="description">Link to your privacy policy page</p>';
    }
    
    public function settings_page() {
        ?>
        <div class="wrap">
            <h1>GDPR Cookie Consent Settings</h1>
            <form method="post" action="options.php">
                <?php
                settings_fields('lgdpr_settings');
                do_settings_sections('lgdpr_settings');
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }
    
    public function enqueue_scripts() {
        if (isset($_COOKIE['lgdpr_accepted'])) return;
        
        wp_enqueue_style('lgdpr-style', plugin_dir_url(__FILE__) . 'lgdpr-style.css', array(), '1.0.0');
        wp_enqueue_script('lgdpr-script', plugin_dir_url(__FILE__) . 'lgdpr-script.js', array(), '1.0.0', true);
        wp_localize_script('lgdpr-script', 'lgdpr_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('lgdpr_nonce')
        ));
    }
    
    public function display_consent_banner() {
        if (isset($_COOKIE['lgdpr_accepted'])) return;
        
        $options = get_option('lgdpr_options');
        $message = isset($options['message']) ? $options['message'] : 'We use cookies to enhance your experience. By continuing to visit this site you agree to our use of cookies.';
        $logo = isset($options['logo']) ? $options['logo'] : '';
        $color = isset($options['color']) ? $options['color'] : '#0073aa';
        $privacy_url = isset($options['privacy_url']) ? $options['privacy_url'] : '';
        
        ?>
        <style>
            .lgdpr-banner { background: #fff; border-top: 3px solid <?php echo esc_attr($color); ?>; }
            .lgdpr-accept { background: <?php echo esc_attr($color); ?>; }
            .lgdpr-accept:hover { opacity: 0.9; }
        </style>
        <div id="lgdpr-consent-banner" class="lgdpr-banner">
            <div class="lgdpr-content">
                <?php if ($logo): ?>
                    <img src="<?php echo esc_url($logo); ?>" alt="Logo" class="lgdpr-logo">
                <?php endif; ?>
                <p class="lgdpr-message">
                    <?php echo esc_html($message); ?>
                    <?php if ($privacy_url): ?>
                        <a href="<?php echo esc_url($privacy_url); ?>" class="lgdpr-link">Learn more</a>
                    <?php endif; ?>
                </p>
                <button id="lgdpr-accept" class="lgdpr-accept">Accept</button>
            </div>
        </div>
        <?php
    }
    
    public function handle_accept() {
        check_ajax_referer('lgdpr_nonce', 'nonce');
        setcookie('lgdpr_accepted', '1', time() + (365 * 24 * 60 * 60), '/');
        wp_send_json_success();
    }
}

new Lightweight_GDPR_Consent();